"use client";
import React, { useState, useEffect } from "react";
 
import Link from "next/link";
import { useSelector } from "react-redux";
 
import { getCookie } from "cookies-next";
import { useRouter } from "next/navigation";
import apiRequest from "@/app/lib/InterceptorAxios/axios";

const roleMapping = {
  2: "Admin",
  3: "Sales",
  4: "Marketing",
};
// Invert the roleMapping object
const roleNameToIdMapping = Object.fromEntries(
  Object.entries(roleMapping).map(([id, name]) => [name, Number(id)])
);

// Function to get role_id from role_name
const getRoleIdFromName = (roleName) => {
  return roleNameToIdMapping[roleName];
};
const fetchUsers = async (brandId) => {
  try {
    
    const data = await apiRequest(`/api/users/brand/${brandId}`);
    return data;
  } catch (error) {
    swal("Error!", error.message, "error");  // Display general error message
  }
 
};

const UserPage = ({ params }) => {
  const { bid } = params;
  const [Brand, setBrand] = useState();
  const router = useRouter();
 

  const user = useSelector((state) => state.user.userInfo);
  const [users, setUsers] = useState([]);
  const [modalOpen, setModalOpen] = useState(false);
  const [UserData, setUserData] = useState();
  const [widgetScriptPath, setWidgetScriptPath] = useState("");
  const [widgetExists, setWidgetExists] = useState(false);
  const token = getCookie("token");
  const [formData, setFormData] = useState({
    brand_id: bid,
    name: "",
    pseudonym: "",
    email: "",
    password: "",
    status: "",
    designation: "",
    role_name: "",
    canCreateUsers: 0,
    canCreateBrand: 0,
      hrm_id:''
  });
  useEffect(() => {
    // Construct the URL using environment variable and dynamic brand ID
    const fetchBrand = async () => {
      try {
        const data = await apiRequest(`/api/brands/${bid}`);

        setBrand(data);
      } catch (error) {
        console.error("Error fetching brand data:", error);
      }
    };

    fetchBrand();
  }, [bid]);
  useEffect(() => {
    const fetchUserInfo = async () => {
      try {
        const data = await apiRequest(`/api/users/${user.userId}`);

        setUserData(data);
        // Handle the response data as needed
      } catch (error) {
        console.error("Error fetching user information:", error);
        // Handle the error as needed
      }
    };
    if (user.userId) {
      fetchUserInfo();
    }
  }, [user.userId]);
  useEffect(() => {
    const fetchData = async () => {
      try {
        const usersData = await fetchUsers(bid);
        setUsers(usersData);
      } catch (error) {
        console.error("Error fetching users:", error.message);
      }
    };
    fetchData();
  }, [bid]);

  const handleChange = (e) => {
    const { name, value } = e.target;

    setFormData({
      ...formData,
      [name]: name === "brand_id" ? parseInt(value, 10) : value, // Convert brandid to integer
    });
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    if (user.userId) {
      const role = getRoleIdFromName(formData.role_name);
      if (role === undefined) {
        alert("Invalid role selected");
        return;
      }
      const created_by = user.userId;
      const company_id = user.company_id;
  
      try {
        await apiRequest('/api/users', 'POST', {
          ...formData, role, created_by, company_id
        });
  
        // Reset form data
        setFormData({
          company_id: "",
          brand_id: bid,
          name: "",
          pseudonym: "",
          email: "",
          password: "",
          status: "",
          role_name: "",
          designation: "",
            hrm_id:''
        });
        setModalOpen(false);
  
        swal("Success!", "User created successfully.", "success");
  
        // Refresh users data
        const usersData = await fetchUsers(bid);
        setUsers(usersData);
  
      } catch (error) {
                if (error.type === 'client' && error.errorData && error.errorData.error === "Email already exists") {
          swal("Oops!", "Email already exists. Please use a different email.", "error");
        } else {
          console.error("Error creating user:", error.message);
          swal("Error!", error.message, "error");  // Display general error message
        }
      }
    } else {
      console.error("User not found");
    }
  };
  
  const fetchWidget = async (brandId) => {
    try {
      const data = await apiRequest(`/api/script/widget`, "POST", {
        brandId: brandId,
      });
      
      return data;
    } catch (error) {
      swal("Error!", error.message, "error");  // Display general error message
    }
    
  };
  //script logic
  useEffect(() => {
    const fetchData = async () => {
      try {
        const widgetData = await fetchWidget(bid);
        if (widgetData.length > 0) {
          setWidgetScriptPath(widgetData[0].file_path);
          setWidgetExists(true);
        } else {
          setWidgetExists(false);
        }
      } catch (error) {
        console.error("Error fetching data:", error.message);
      }
    };

    fetchData();
  }, [bid]);
  const handleCopyScript = async () => {
    if (widgetScriptPath.length > 0) {
      const script = `<script async src="${process.env.NEXT_PUBLIC_API_URL2}${widgetScriptPath}"></script>`;

      try {
        // Attempt to use the Clipboard API
        await navigator.clipboard.writeText(script);
        
        swal("Success!", "Script copied to clipboard!", "success");
      } catch (err) {
        console.error(
          "Failed to copy using Clipboard API, trying fallback:",
          err
        );

        // Fallback method using a temporary textarea
        const tempTextArea = document.createElement("textarea");
        tempTextArea.value = script;
        document.body.appendChild(tempTextArea);
        tempTextArea.select();

        try {
          const successful = document.execCommand("copy");
          if (successful) {
            alert("Script copied to clipboard!");
          } else {
            alert("copy failed.");
          }
        } catch (fallbackErr) {
          console.error("Fallback copy error:", fallbackErr);
          alert("Failed to copy script to clipboard.");
        } finally {
          document.body.removeChild(tempTextArea); // Clean up
        }
      }
    } else {
      alert("No script available to copy.");
    }
  };

  const handleGenerateScript = async () => {
    try {
      const res = await fetch(
        `${process.env.NEXT_PUBLIC_API_URL2}/api/generate-widget`,
        {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
          },
          body: JSON.stringify({ brandId: bid, companyId: user.company_id }),
        }
      );

      if (res.ok) {
        const widgetData = await res.json();

        setWidgetScriptPath(widgetData.filePath);
        setWidgetExists(true);
      
        swal("Success!", "Widget created successfully!", "success");
      } else {
        alert("Error creating widget");
      }
    } catch (error) {
      console.error("Error creating widget:", error.message);
    }
  };
  return (
    <>
      <div className="container-fluid top-bar">
        <div className="left-div">
          <h2>{Brand?.name}: Users</h2>
        </div>
        <div className="right-div"></div>
      </div>
      <div className="container-fluid company-header">
        <div className="comp-header-row">
        
          {[1, 2, 4].includes(user.role) && (
            <>
              {widgetExists ? (
                <button className="btn mr-2 me-2" onClick={handleCopyScript}>
                  Copy Script
                </button>
              ) : (
                <button className="btn" onClick={handleGenerateScript}>
                  Generate Script
                </button>
              )}
              {UserData?.canCreateUsers === 1 && (
                <button className="btn" onClick={() => setModalOpen(true)}>
                  Auto Replies
                </button>
              )} {UserData?.canCreateUsers === 1 && (
                <button className="btn" onClick={() => setModalOpen(true)}>
                  Add User
                </button>
              )}
            </>
          )}
        </div>
      </div>
      <div className="container-fluid">
        <table className="table table-striped mt-4">
          <thead>
            <tr>
              <th>Real Name</th>
              <th>Pseudonym</th>
              <th>Designation</th>
              <th>Email</th>
              {/* Conditionally render the following headers based on user.role */}
              {[1, 2, 4].includes(user?.role) && (
                <>
                  <th>Role</th>
                  <th>Created At</th>
                  <th>Status</th>
                  <th>Actions</th>
                </>
              )}
            </tr>
          </thead>
          <tbody>
            {users.length === 0 ? (
              <tr>
                <td colSpan="7">No Record found</td>
              </tr>
            ) : (
              users.map((userz) => (
                <tr key={userz?.id}>
                  <td>{userz?.name || "N/A"}</td>
                  <td>{userz?.pseudonym || "N/A"}</td>
                  <td>{userz?.designation || "N/A"}</td>
                  <td>{userz?.email || "N/A"}</td>
                  {/* Conditionally render the following cells based on user.role */}
                  {[1, 2, 4].includes(user?.role) && (
                    <>
                      <td>{roleMapping[userz?.role_id] || "Unknown"}</td>
                      <td>
                        {userz?.created_at
                          ? new Date(userz.created_at).toLocaleDateString()
                          : "N/A"}
                      </td>
                      <td>{userz?.status === 1 ? "Active" : "Inactive"}</td>
                      <td>
                     
                      </td>
                    </>
                  )}
                </tr>
              ))
            )}
          </tbody>
        </table>
      </div>

      {modalOpen && (
        <div className="modal-overlay-comp">
          <div className="modal-brand">
            <div className="modal-header-comp">
              <h5 className="modal-title">Add User</h5>
              <button
                type="button"
                className="modal-close"
                onClick={() => setModalOpen(false)}
              >
                &times;
              </button>
            </div>
            <div className="modal-body">
              <form onSubmit={handleSubmit}>
                <div className="form-group-row-2">
                  <div className="form-group">
                    <label htmlFor="name">Real Name:</label>
                    <input
                      className="form-control"
                      type="text"
                      id="name"
                      name="name"
                      value={formData.name}
                      onChange={handleChange}
                      required
                    />
                  </div>
                  <div className="form-group">
                    <label htmlFor="pseudonym">Pseudonym:</label>
                    <input
                      className="form-control"
                      type="text"
                      id="pseudonym"
                      name="pseudonym"
                      value={formData.pseudonym}
                      onChange={handleChange}
                      required
                    />
                  </div>
                </div>
                <div className="form-group-row-2">
                  <div className="form-group">
                    <label htmlFor="email">Email:</label>
                    <input
                      className="form-control"
                      type="email"
                      id="email"
                      name="email"
                      value={formData.email}
                      onChange={handleChange}
                      required
                    />
                  </div>
                  <div className="form-group">
                    <label htmlFor="password">Password:</label>
                    <input
                      className="form-control"
                      type="password"
                      id="password"
                      name="password"
                      value={formData.password}
                      onChange={handleChange}
                      required
                    />
                  </div>
                </div>
                <div className="form-group">
                    <label htmlFor="designation">HRM ID:</label>
                    <input
                      type="text"
                      className="form-control"
                      id="hrm_id"
                      name="hrm_id"
                      value={formData.hrm_id}
                      onChange={handleChange}
                      required
                    />
                  </div>
                <div className="form-group">
                  <label htmlFor="designation">Designation:</label>
                  <input
                    type="text"
                    className="form-control"
                    id="designation"
                    name="designation"
                    value={formData.designation}
                    onChange={handleChange}
                    required
                  />
                </div>
                <div className="form-group">
                  <label htmlFor="status">Status:</label>
                  <select
                    className="form-control"
                    id="status"
                    name="status"
                    value={formData.status}
                    onChange={handleChange}
                    required
                  >
                    <option value="">Select a status</option>
                    <option value="1">Active</option>
                    <option value="0">Inactive</option>
                  </select>
                </div>
                <div className="form-group">
                  <label htmlFor="role_name">Role:</label>
                  <select
                    className="form-control"
                    id="role_name"
                    name="role_name"
                    value={formData.role_name}
                    onChange={handleChange}
                    required
                  >
                    <option value="">Select a role</option>

                    <option value="Admin">Admin</option>
                    <option value="Sales" selected>Sales</option>
                    <option value="Marketing">Marketing</option>
             
                  </select>
                </div>
                <div className="form-group">
                  <label htmlFor="canCreateUsers">Can Create Users</label>
                  <select
                    className="form-control"
                    id="canCreateUsers"
                    name="canCreateUsers"
                    value={formData.canCreateUsers}
                    onChange={handleChange}
                    required
                  >
                    <option value={0}>No</option>
                    <option value={1}>Yes</option>
                  </select>
                </div>
                <div className="form-group">
                  <label htmlFor="canCreateBrand">Can Create Brand</label>
                  <select
                    className="form-control"
                    id="canCreateBrand"
                    name="canCreateBrand"
                    value={formData.canCreateBrand}
                    onChange={handleChange}
                    required
                  >
                    <option value={0}>No</option>
                    <option value={1}>Yes</option>
                  </select>
                </div>
                <button
                  type="submit"
                  className="btn submit-btn p-btn mt-3 w-100"
                >
                  Submit
                </button>
              </form>
            </div>
          </div>
        </div>
      )}
    </>
  );
};

export default UserPage;
